const express = require('express');
const router = express.Router();
const axios = require('axios');
const { query, collegeQuery } = require('../../config/db');
const { generateOTP, checkSMSStatus, sendSMS } = require('../verifiotp');

const sendWebhookToCRM = async (data) => {
    const webhookUrl = `${process.env.CRM_WEBHOOK_URL}leads`;
    const response = await axios.post(webhookUrl, data);
    console.log('webhook response:', response.data);
};

const sendWebhookToCRMForEnquiry = async (data) => {
    const webhookUrl = `${process.env.CRM_WEBHOOK_URL}enquiry`;
    const response = await axios.post(webhookUrl, data);
    console.log('webhook response:', response.data);
};

router.post('/register', async (req, res) => {
    console.log('in register');

    // Extract form data
    let {
        courses,
        student_name,
        student_email,
        mobile_number,
        degrees,
        specialization,
        yop,
        perc_marks,
        college_state,
        college_name,
        college_code,
        college_name_full,
        other_colledge_name = null,
        trng_centre,
        utm_source = null,
        utm_medium = null,
        utm_campaign = null
    } = req.body;

    // Map to expected field names for backward compatibility
    const name = student_name;
    const email = student_email;

    console.log('student registration data:', {
        courses, name, email, mobile_number, degrees, specialization, college_code, college_name_full,
        yop, perc_marks, college_state, college_name, other_colledge_name,
        trng_centre, utm_source, utm_medium, utm_campaign
    });

    // TODO: Validate the data
    if (!name || !email || !mobile_number || !courses) {
        return res.status(400).json({
            success: false,
            error: 1,
            message: 'Missing required fields: name, email, mobile_number, course'
        });
    }

    try {
        // get entrance code
        const entranceCodeQuery = await query('SELECT entranceid FROM entrance WHERE course_name = ? AND active = 1 and registrationstart <= NOW() order by id desc limit 1', [courses]);
        const entranceCode = entranceCodeQuery[0]?.entranceid || process.env.ENTRANCE_CODE;

        // check if user already exists
        const existingUser = await query('SELECT * FROM student_registrations WHERE email = ? AND mobile = ? and course_name = ? and entranceid = ? limit 1', [email, mobile_number, courses, entranceCode]);
        if (existingUser.length > 0) {
            // check if mobile number is already verified or not
            const verifiedMobile = await query('SELECT * FROM verification_otps WHERE mobile = ? AND verified = true', [mobile_number]);
            let mobileVerified = false;
            if (verifiedMobile.length > 0) {
                mobileVerified = true;
                //mobile number is already verified show options to generate exam link or show the exam result.
                //get the user data from the database
                const url = "https://vectorindia.org/exam/api/registrations/";
                const examResponse = await axios.post(url, {
                    student_name: name,
                    student_email: email,
                    student_phone: mobile_number,
                    unique_id: existingUser[0].id,
                    examId: courses == "Embedded Systems" ? 1 : 2
                });
                console.log('examResponse:', examResponse.data);
                if (examResponse.data?.examLink) {
                    await query('UPDATE student_registrations SET exam_invitation_response = ? WHERE id = ?', [JSON.stringify(examResponse.data), existingUser[0].id]);
                    existingUser[0].exam_invitation_response = examResponse.data;
                }
            }
            //TODO: Send data to CRM student already registered may be interseted in other courses.
            // send webhook to CRM
            console.log('sending webhook to CRM:', {
                courses, student_name: name, student_email: email, mobile_number: mobile_number, degrees, specialization, collegeCode: college_code, collegeName: college_name_full, entranceid: entranceCode,
                yop, perc_marks, college_state, college_name, other_colledge_name,
                trng_centre, utm_source, utm_medium, utm_campaign, exam_invitation_response: existingUser[0].exam_invitation_response, mobileVerified: mobileVerified
            });
            await sendWebhookToCRM({
                courses, student_name: name, student_email: email, mobile_number: mobile_number, degrees, specialization, collegeCode: college_code, collegeName: college_name_full, entranceid: entranceCode,
                yop, perc_marks, college_state, college_name, other_colledge_name,
                trng_centre, utm_source, utm_medium, utm_campaign, exam_invitation_response: existingUser[0].exam_invitation_response, mobileVerified: mobileVerified
            });


            if (verifiedMobile.length === 0) {
                //send otp to mobile number
                const otp = generateOTP();
                const hashId = require('crypto').randomBytes(16).toString('hex');
                const sendResult = await sendSMS(mobile_number, otp);
                const messageId = (sendResult.success) ? sendResult.messageId : '';
                await query('INSERT INTO verification_otps (hash_id, mobile, otp, method, created_at, status, student_id) VALUES (?, ?, ?, ?, NOW(), ?, ?)', [hashId, mobile_number, otp, 'sms', messageId, existingUser[0].id]);
                return res.status(200).json({
                    success: true,
                    error: 0,
                    verified: false,
                    data: {
                        hashId: hashId
                    },
                    message: 'OTP sent successfully'
                });
            }
            return res.status(200).json({ success: true, error: 0, message: 'User already registered', data: existingUser[0], verified: 1 });
        }

        //check if other college name is not empty
        if (other_colledge_name === '' || other_colledge_name === null) {
            // insert a new college in the database
            const collegeResult = await query('INSERT INTO college (state_id, college_name) VALUES (?, ?)', [college_state, other_colledge_name]);
            college_name = parseInt(collegeResult.insertId);
        }

        // Insert into student_registrations table (matching the schema you showed)
        const result = await query(`
            INSERT INTO student_registrations (
                course_name, name, email, mobile, pgugothers, degree, yop, 
                percentage, specialization, c_state, c_name, trng_centre,
                utm_source, utm_medium, utm_campaign, entranceid
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        `, [
            courses,                    // course_name
            name,                       // name
            email,                      // email
            mobile_number,              // mobile
            degrees,                    // pgugothers (UG/PG/Others)
            degrees,                    // degree (same as pgugothers for now)
            yop,                        // yop
            parseFloat(perc_marks),     // percentage (convert to number)
            specialization,             // specialization
            college_state,              // c_state
            college_name,               // c_name
            trng_centre,                // trng_centre
            utm_source,                 // utm_source
            utm_medium,                 // utm_medium
            utm_campaign,               // utm_campaign
            entranceCode                // entranceid
        ]);

        console.log('Registration successful:', result);
        const srtudentRecordId = parseInt(result.insertId);

        console.log('before sending webhook to CRM:', {
            courses, student_name: name, student_email: email, mobile_number: mobile_number, degrees, specialization, collegeCode: college_code, collegeName: college_name_full, entranceid: entranceCode,
            yop, perc_marks, college_state, college_name, other_colledge_name,
            trng_centre, utm_source, utm_medium, utm_campaign
        });

        //send otp to mobile number
        //check if mobile number is already verified or sent otp in last 5 minutes
        const verifiedMobile = await query('SELECT * FROM verification_otps WHERE mobile = ? AND verified = true', [mobile_number]);
        let mobileVerified = false;
        if (verifiedMobile.length > 0) {
            mobileVerified = true;
        }

        if (verifiedMobile.length > 0) {
            //mobile number is already verified show options to generate exam link or show the exam result.
            //get the user data from the database
            const url = "https://vectorindia.org/exam/api/registrations/";
            const examResponse = await axios.post(url, {
                student_name: name,
                student_email: email,
                student_phone: mobile_number,
                unique_id: srtudentRecordId,
                examId: courses == "Embedded Systems" ? 1 : 2
            });
            console.log('examResponse:', examResponse.data);
            if (examResponse.data?.examLink) {
                await query('UPDATE student_registrations SET exam_invitation_response = ? WHERE id = ?', [JSON.stringify(examResponse.data), srtudentRecordId]);
            }
            //send webhook to CRM
            await sendWebhookToCRM({
                courses, student_name: name, student_email: email, mobile_number: mobile_number, degrees, specialization, collegeCode: college_code, collegeName: college_name_full, entranceid: entranceCode,
                yop, perc_marks, college_state, college_name, other_colledge_name,
                trng_centre, utm_source, utm_medium, utm_campaign, exam_invitation_response: examResponse.data, mobileVerified: mobileVerified
            });

            return res.status(200).json({
                success: true,
                verified: true,
                error: 0,
                data: {
                    id: srtudentRecordId,
                    exam_invitation_response: examResponse.data,
                    entranceCode: entranceCode
                },
                message: 'Mobile number is already verified'
            });
        }
        // Send data to CRM
        await sendWebhookToCRM({
            courses, student_name: name, student_email: email, mobile_number: mobile_number, degrees, specialization, collegeCode: college_code, collegeName: college_name_full, entranceid: entranceCode,
            yop, perc_marks, college_state, college_name, other_colledge_name,
            trng_centre, utm_source, utm_medium, utm_campaign, mobileVerified: mobileVerified
        });
        const otp = generateOTP();
        const hashId = require('crypto').randomBytes(16).toString('hex');
        const sendResult = await sendSMS(mobile_number, otp);
        const messageId = (sendResult.success) ? sendResult.messageId : '';
        await query('INSERT INTO verification_otps (hash_id, mobile, otp, method, created_at, status, student_id) VALUES (?, ?, ?, ?, NOW(), ?, ?)', [hashId, mobile_number, otp, 'sms', messageId, srtudentRecordId]);

        res.json({
            success: true,
            error: 0,
            verified: false,
            data: {
                hashId: hashId
            },
            message: 'OTP sent successfully'
        });

    } catch (error) {
        console.error('Registration error:', error);
        res.status(500).json({
            success: false,
            error: 1,
            message: 'Registration failed',
            details: error.message
        });
    }
});




router.post('/check-sms-status', async (req, res) => {
    const { messageId } = req.body;
    const response = await checkSMSStatus(messageId);
    res.json(response);
});

router.post('/resend-otp', async (req, res) => {
    const { mobile_number } = req.body;
    //max otp message per day is 3
    const otpMessage = await query('SELECT * FROM verification_otps WHERE mobile = ? AND created_at > NOW() - INTERVAL 1 DAY order by id desc', [mobile_number]);
    if (otpMessage.length >= 3) {
        return res.status(400).json({
            error: 1,
            success: false,
            message: 'Max OTP messages per day reached'
        });
    }
    if (otpMessage.length === 0) {
        return res.status(400).json({
            error: 1,
            success: false,
            message: 'OTP not found'
        });
    }
    const srtudentRecordId = otpMessage[0].student_id;
    const otp = generateOTP();
    const hashId = require('crypto').randomBytes(16).toString('hex');
    const sendResult = await sendSMS(mobile_number, otp);
    const messageId = (sendResult.success) ? sendResult.messageId : '';
    await query('INSERT INTO verification_otps (hash_id, mobile, otp, method, created_at, status, student_id) VALUES (?, ?, ?, ?, NOW(), ?, ?)', [hashId, mobile_number, otp, 'sms', messageId, srtudentRecordId]);

    return res.status(200).json({
        success: true,
        error: 0,
        verified: false,
        data: {
            hashId: hashId
        },
        message: 'OTP sent successfully'
    });
});


const verifyOTP = async (hashId, otp) => {
    const response = await query('SELECT * FROM verification_otps WHERE hash_id = ?', [hashId]);
    if (response[0].failed_attempts >= 5) {
        return {
            success: false,
            error: 1,
            message: 'Max OTP attempts reached'
        };
    }
    if (response.length > 0) {
        if (response[0].otp === otp) {
            await query('UPDATE verification_otps SET verified = true, verified_at = NOW() WHERE id = ?', [response[0].id]);
            return {
                success: true,
                message: 'OTP verified successfully',
                data: response[0]
            };
        } else {
            await query('UPDATE verification_otps SET failed_attempts = failed_attempts + 1 WHERE hash_id = ?', [hashId]);
            return {
                success: false,
                error: 1,
                message: 'Invalid OTP'
            };
        }
    } else {
        await query('UPDATE verification_otps SET failed_attempts = failed_attempts + 1 WHERE hash_id = ?', [hashId]);
        return {
            success: false,
            error: 1,
            message: 'Invalid OTP'
        };
    }
};

router.post('/verify-otp', async (req, res) => {
    const { hashId, otp } = req.body;
    const response = await verifyOTP(hashId, otp);
    console.log('response:', response);
    if (response.success) {
        //get the user data from the database
        const userData = await query('SELECT * FROM student_registrations WHERE mobile = ? order by id desc limit 1', [response.data.mobile]);
        console.log('userData:', userData);
        const url = "https://vectorindia.org/exam/api/registrations/";
        const examResponse = await axios.post(url, {
            student_name: userData[0].name,
            student_email: userData[0].email,
            student_phone: userData[0].mobile,
            unique_id: parseInt(userData[0].id),
            examId: userData[0].course_name == "Embedded Systems" ? 1 : 2
        });
        console.log('examResponse:', examResponse.data);
        if (examResponse.data.success) {
            await query('UPDATE student_registrations SET exam_invitation_response = ? WHERE id = ?', [examResponse.data.data, userData[0].id]);
            userData[0].exam_invitation_response = examResponse.data.data;
        }
        //send webhook to CRM
        let collegeDetails = await collegeQuery('SELECT * FROM collegedirectory WHERE id = ?', [userData[0].c_name]);
        if (collegeDetails.length === 0) {
            collegeDetails = await query('SELECT * FROM college WHERE id = ?', [userData[0].c_name]);
        }
        console.log('before sending webhook to CRM:', {
            courses: userData[0].course_name, student_name: userData[0].name, student_email: userData[0].email, mobile_number: userData[0].mobile, degrees: userData[0].degree, specialization: userData[0].specialization, collegeCode: collegeDetails[0].college_code, collegeName: collegeDetails[0].college_name, entranceid: userData[0].entranceid, yop: userData[0].yop, perc_marks: userData[0].percentage, college_state: userData[0].c_state, trng_centre: userData[0].trng_centre, utm_source: userData[0].utm_source, utm_medium: userData[0].utm_medium, utm_campaign: userData[0].utm_campaign, exam_invitation_response: userData[0].exam_invitation_response
        });
        await sendWebhookToCRM({
            courses: userData[0].course_name, student_name: userData[0].name, student_email: userData[0].email, mobile_number: userData[0].mobile, degrees: userData[0].degree, specialization: userData[0].specialization, collegeCode: collegeDetails[0].college_code, collegeName: collegeDetails[0].college_name, entranceid: userData[0].entranceid, yop: userData[0].yop, perc_marks: userData[0].percentage, college_state: userData[0].c_state, trng_centre: userData[0].trng_centre, utm_source: userData[0].utm_source, utm_medium: userData[0].utm_medium, utm_campaign: userData[0].utm_campaign, exam_invitation_response: userData[0].exam_invitation_response, mobileVerified: true
        });
        return res.status(200).json({
            success: true,
            message: 'OTP verified successfully',
            data: response.data,
            studentData: userData[0]
        });
    } else {
        return res.status(400).json({
            success: false,
            error: 1,
            message: response.message
        });
    }
});

router.post('/enquiry/resend-otp', async (req, res) => {
    const { mobile } = req.body;

    if (!mobile) {
        return res.status(400).json({
            error: 1,
            success: false,
            message: 'Mobile number is required'
        });
    }

    try {
        //max otp message per day is 3
        const otpMessage = await query('SELECT * FROM verification_otps WHERE mobile = ? AND created_at > NOW() - INTERVAL 1 DAY order by id desc', [mobile]);
        if (otpMessage.length >= 3) {
            return res.status(400).json({
                error: 1,
                success: false,
                message: 'Max OTP messages per day reached'
            });
        }
        if (otpMessage.length === 0) {
            return res.status(400).json({
                error: 1,
                success: false,
                message: 'No enquiry found for this mobile number'
            });
        }
        const srtudentRecordId = otpMessage[0].student_id;
        const otp = generateOTP();
        const hashId = require('crypto').randomBytes(16).toString('hex');
        const sendResult = await sendSMS(mobile, otp);
        const messageId = (sendResult.success) ? sendResult.messageId : '';
        await query('INSERT INTO verification_otps (hash_id, mobile, otp, method, created_at, status, student_id) VALUES (?, ?, ?, ?, NOW(), ?, ?)', [hashId, mobile, otp, 'sms', messageId, srtudentRecordId]);

        return res.status(200).json({
            success: true,
            error: 0,
            message: 'OTP sent successfully'
        });
    } catch (error) {
        console.error('Resend OTP error:', error);
        return res.status(500).json({
            error: 1,
            success: false,
            message: 'Failed to resend OTP'
        });
    }
});

router.post('/enquiry/verify-otp', async (req, res) => {
    const { hashId, otp } = req.body;
    const response = await verifyOTP(hashId, otp);
    console.log('response:', response);
    if (response.success) {
        //get the user data from the database
        const existingUser = await query('SELECT * FROM enquiry WHERE id = ? ', [response.data.student_id]);
        //TODO: send email to the user

        //send webhook to CRM
        console.log('sending webhook to CRM for enquiry:', {
            name: existingUser[0].full_name, email: existingUser[0].email, mobile: existingUser[0].mobile, course_interestfor: existingUser[0].interested, trng_centre: existingUser[0].trng_centre, enquiry: existingUser[0].enquiry, utm_source: existingUser[0].utm_source, utm_medium: existingUser[0].utm_medium, utm_campaign: existingUser[0].utm_campaign, type: existingUser[0].type, camefrom: existingUser[0].camefrom, mobileVerified: true
        });
        await sendWebhookToCRMForEnquiry({
            name: existingUser[0].full_name, email: existingUser[0].email, mobile: existingUser[0].mobile, course_interestfor: existingUser[0].interested, trng_centre: existingUser[0].trng_centre, enquiry: existingUser[0].enquiry, utm_source: existingUser[0].utm_source, utm_medium: existingUser[0].utm_medium, utm_campaign: existingUser[0].utm_campaign, type: existingUser[0].type, camefrom: existingUser[0].camefrom, mobileVerified: true
        });
        return res.status(200).json({
            success: true,
            error: 0,
            message: 'Thank you for submitting the details. Our team will contact you soon'
        });
    } else {
        return res.status(200).json({
            success: false,
            error: 1,
            message: response.message
        });
    }
});

router.post('/enquiry/save', async (req, res) => {

    //TODO: validate the request should not come from the same ip address in last 5 minutes
    // some kind of security check to prevent spam

    //name=Rudraraju+Sita+Rama+Anjaneya+Raju&email=rudhrarajus%40gmail.com&mobile=9989434569&course_interestfor=Full+Stack+Development&trng_centre=Hyderabad&enquiry=testing&utm_source=&utm_medium=&utm_campaign=&type=&camefrom=Triggered+by+Quick+Enquiry+%0D%0A+++++++++from+thank-you.html
    // TODO: validate the fields of the request
    const { name, email, mobile, course_interestfor, trng_centre, enquiry, utm_campaign, camefrom, designation = null, company_name = null } = req.body;

    let utm_medium = req.body.utm_medium;
    if (utm_medium === null || utm_medium === undefined || utm_medium === '') {
        utm_medium = 'Contact Form';
    }

    let utm_source = req.body.utm_source;
    if (utm_source === null || utm_source === undefined || utm_source === '') {
        utm_source = camefrom.slice(0, 50);
    }

    let type = req.body.type;
    if (type === null || type === undefined || type === '') {
        type = 'enquiry';
    }
    let enquiryDetails = '';
    if (type && type == 'corp-enquiry') {
        if (designation) {
            enquiryDetails += 'Designation: ' + designation + '\n\n';
        }
        if (company_name) {
            enquiryDetails += 'Company Name: ' + company_name + '\n\n';
        }
    }
    enquiryDetails += '\n\n';
    enquiryDetails += enquiry + '\n\n';

    try {

        // check if user already submitted the enquiry
        const existingUser = await query('SELECT * FROM enquiry WHERE email = ? AND mobile = ? AND interested = ? AND enquiry = ? limit 1', [email, mobile, course_interestfor, enquiryDetails]);
        if (existingUser.length > 0) {
            // check if mobile number is already verified or not
            const verifiedMobile = await query('SELECT * FROM verification_otps WHERE mobile = ? AND verified = true', [mobile]);

            const isMobileVerified = (verifiedMobile.length > 0) ? true : false;
            // Send data to CRM student already registered may be interseted in other courses.
            console.log('sending webhook to CRM for enquiry:', {
                name: existingUser[0].full_name, email: existingUser[0].email, mobile: existingUser[0].mobile, course_interestfor: existingUser[0].interested, trng_centre: existingUser[0].trng_centre, enquiry: existingUser[0].enquiry, utm_source: existingUser[0].utm_source, utm_medium: existingUser[0].utm_medium, utm_campaign: existingUser[0].utm_campaign, type: existingUser[0].type, camefrom: existingUser[0].camefrom, mobileVerified: isMobileVerified, designation, company_name
            });
            await sendWebhookToCRMForEnquiry({
                name: existingUser[0].full_name, email: existingUser[0].email, mobile: existingUser[0].mobile, course_interestfor: existingUser[0].interested, trng_centre: existingUser[0].trng_centre, enquiry: existingUser[0].enquiry, utm_source: existingUser[0].utm_source, utm_medium: existingUser[0].utm_medium, utm_campaign: existingUser[0].utm_campaign, type: existingUser[0].type, camefrom: existingUser[0].camefrom, mobileVerified: isMobileVerified, designation, company_name
            });

            if (verifiedMobile.length === 0) {
                //send otp to mobile number
                const otp = generateOTP();
                const hashId = require('crypto').randomBytes(16).toString('hex');
                const sendResult = await sendSMS(mobile, otp);
                const messageId = (sendResult.success) ? sendResult.messageId : '';
                await query('INSERT INTO verification_otps (hash_id, mobile, otp, method, created_at, status, student_id) VALUES (?, ?, ?, ?, NOW(), ?, ?)', [hashId, mobile, otp, 'sms', messageId, existingUser[0].id]);
                return res.status(200).json({
                    success: true,
                    error: 0,
                    verified: false,
                    otp_sent: true,
                    data: {
                        hashId: hashId
                    },
                    message: 'Please validate the OTP'
                });
            }
            return res.status(200).json({
                success: true,
                error: 0,
                message: 'Thank you for submitting the details. Our team will contact you soon'
            });
        }
        const response = await query('INSERT INTO enquiry (full_name, email, mobile, enquiry,type,interested,camefrom,trng_centre,utm_source,utm_medium,utm_campaign) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)', [name, email, mobile, enquiryDetails, type, course_interestfor, camefrom, trng_centre, utm_source, utm_medium, utm_campaign]);


        console.log('Enquiry Saved successful:', response);
        const srtudentRecordId = parseInt(response.insertId);

        //send otp to mobile number
        //check if mobile number is already verified or sent otp in last 5 minutes
        const verifiedMobile = await query('SELECT * FROM verification_otps WHERE mobile = ? AND verified = true', [mobile]);
        const isMobileVerified = (verifiedMobile.length > 0) ? true : false;
        console.log('sending webhook to CRM for enquiry:', {
            name, email, mobile, course_interestfor, trng_centre, enquiry: enquiryDetails, utm_source, utm_medium, utm_campaign, type, camefrom, mobileVerified: isMobileVerified, designation, company_name
        });
        await sendWebhookToCRMForEnquiry({
            name, email, mobile, course_interestfor, trng_centre, enquiry: enquiryDetails, utm_source, utm_medium, utm_campaign, type, camefrom, mobileVerified: isMobileVerified, designation, company_name
        });
        if (verifiedMobile.length > 0) {
            //mobile number is already verified show options to generate exam link or show the \
            return res.status(200).json({
                success: true,
                error: 0,
                message: 'Thank you for submitting the details. Our team will contact you soon'
            });
        }

        const otp = generateOTP();
        const hashId = require('crypto').randomBytes(16).toString('hex');
        const sendResult = await sendSMS(mobile, otp);
        const messageId = (sendResult.success) ? sendResult.messageId : '';
        await query('INSERT INTO verification_otps (hash_id, mobile, otp, method, created_at, status, student_id) VALUES (?, ?, ?, ?, NOW(), ?, ?)', [hashId, mobile, otp, 'sms', messageId, srtudentRecordId]);
        //'error' => 0,
        // 'otp_sent' => 'yes',
        // 'message' => 'Please validate the OTP',
        // 'token' => csrf_hash()
        res.json({
            success: true,
            error: 0,
            verified: false,
            otp_sent: true,
            data: {
                hashId: hashId
            },
            message: 'Please validate the OTP'
        });

    } catch (error) {
        console.error('enquiry error:', error);
        res.status(500).json({
            success: false,
            error: 1,
            message: 'Enquiry failed',
            details: error.message
        });
    }
});

module.exports = router;    