const router = require("express").Router();
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const { vtdbquery } = require('../config/db');
const { auth, checkRole } = require('./authMiddleware');
const nodemailer = require('nodemailer');

// Create email transporter
const transporter = nodemailer.createTransport({
    host: process.env.SMTP_HOST,
    port: process.env.SMTP_PORT,
    secure: false, // false for port 587 (STARTTLS)
    requireTLS: true, // Force TLS
    auth: {
        user: process.env.SMTP_USER,
        pass: process.env.SMTP_PASSWORD
    },
    tls: {
        ciphers: 'SSLv3',
        rejectUnauthorized: true // Use this only if you're having certificate issues
    }
});

// Store OTPs temporarily (in production, use Redis or similar)
const otpStore = new Map();

// Generate OTP
function generateOTP() {
    return Math.floor(100000 + Math.random() * 900000).toString();
}

// Helper function to get client IP
const getClientIP = (req) => {
    return req.ip ||
        (req.socket && req.socket.remoteAddress) ||
        'unknown';
};

// User Registration Route
router.post("/register", async (req, res) => {
    try {
        const { email, password, name } = req.body;

        // Validation
        if (!email || !password || !name) {
            return res.status(400).json({
                errorMessage: "Please enter all required fields.",
            });
        }

        if (password.length < 6) {
            return res.status(400).json({
                errorMessage: "Password must be at least 6 characters long.",
            });
        }

        // Check if user already exists
        const existingUser = await vtdbquery("SELECT * FROM userstable WHERE email = ?", [email]);
        if (existingUser.length > 0) {
            return res.status(400).json({
                errorMessage: "An account with this email already exists.",
            });
        }

        // Hash the password
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(password, salt);

        // Create the user
        const result = await vtdbquery(
            "INSERT INTO userstable (email, password, status) VALUES (?, ?, ?)",
            [email, hashedPassword, 'pending']
        );

        const userId = result.insertId;

        // Create user profile
        await vtdbquery(
            "INSERT INTO user_profiles (user_id, full_name) VALUES (?, ?)",
            [userId, name]
        );

        // Assign default role (FrontOffice)
        const frontOfficeRole = await vtdbquery("SELECT id FROM roles WHERE name = ?", ["FrontOffice"]);
        if (frontOfficeRole.length > 0) {
            await vtdbquery(
                "INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)",
                [userId, frontOfficeRole[0].id]
            );
        }

        res.status(201).json({
            success: true,
            message: "User registered successfully. Please wait for admin approval.",
        });
    } catch (err) {
        console.error("Registration error:", err);
        res.status(500).json({
            errorMessage: "An error occurred during registration.",
        });
    }
});

// User Login Route
router.post("/login", async (req, res) => {
    let userId = null; // Declare userId at the top of the function
    let loginStatus = 'failed';
    let failureReason = null;

    try {
        const { email, password } = req.body;
        const ipAddress = getClientIP(req);
        const userAgent = req.headers['user-agent'];

        // validation
        if (!email || !password) {
            failureReason = "Missing required fields";
            // Don't log to audit table for validation errors
            return res.status(400).json({
                errorMessage: "Please enter all required fields.",
            });
        }

        // get user account
        const result = await vtdbquery("SELECT * FROM userstable WHERE email = ?", [email]);
        if (result.length === 0) {
            failureReason = "Invalid credentials";
            // Don't log to audit table for invalid credentials
            return res.status(401).json({
                errorMessage: "Wrong email or password.",
            });
        }

        const user = result[0];
        userId = user.id;

        // Check if user is active
        if (user.status !== 'active') {
            failureReason = "Account not active";
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, ?, ?)",
                [userId, ipAddress, userAgent, loginStatus, failureReason]
            );
            return res.status(401).json({
                errorMessage: "Your account is not active. Please contact an administrator.",
            });
        }

        let passwordHash = user.password;
        let userpassword = password;
        if (user.password === "test") {
            const salt = await bcrypt.genSalt();
            passwordHash = await bcrypt.hash("vector369", salt);
            userpassword = "vector369";
            await vtdbquery("UPDATE userstable SET password = ? WHERE email = ?", [passwordHash, email]);
        }

        const correctPassword = await bcrypt.compare(
            userpassword,
            passwordHash
        );

        if (!correctPassword) {
            failureReason = "Invalid credentials";
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, ?, ?)",
                [userId, ipAddress, userAgent, loginStatus, failureReason]
            );
            return res.status(401).json({
                errorMessage: "Wrong email or password.",
            });
        }

        // Update last login time and log successful login
        loginStatus = 'success';
        await Promise.all([
            vtdbquery("UPDATE userstable SET last_login = NOW() WHERE id = ?", [user.id]),
            vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, ?, ?)",
                [userId, ipAddress, userAgent, loginStatus, null]
            )
        ]);

        // Get user roles
        const userRoles = await vtdbquery(`
            SELECT r.name 
            FROM user_roles ur
            JOIN roles r ON ur.role_id = r.id
            WHERE ur.user_id = ?
        `, [user.id]);

        // Get user profile
        const userProfile = await vtdbquery("SELECT * FROM user_profiles WHERE user_id = ?", [user.id]);

        const token = jwt.sign(
            {
                email: user.email,
                id: user.id,
                name: userProfile.length > 0 ? userProfile[0].full_name : null,
                roles: userRoles.map(role => role.name),
                mobile: userProfile.length > 0 ? userProfile[0].mobile : null,
                pbx_extension: user.pbx_extension,
                pbx_port: user.pbx_port
            },
            process.env.JWT_SECRET,
            { expiresIn: '24h' }
        );

        // Set secure HTTP-only cookie
        res.cookie('token', token, {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'strict',
            maxAge: 24 * 60 * 60 * 1000 // 24 hours
        });

        res.json({
            success: true,
            token,
            user: {
                id: user.id,
                email: user.email,
                name: userProfile.length > 0 ? userProfile[0].full_name : null,
                roles: userRoles.map(role => role.name),
                mobile: userProfile.length > 0 ? userProfile[0].mobile : null,
                pbx_extension: user.pbx_extension,
                pbx_port: user.pbx_port
            }
        });
    } catch (err) {
        console.error("Login error: ", err);
        // Only log to audit if we have a userId
        if (userId) {
            try {
                await vtdbquery(
                    "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                    [userId, getClientIP(req), req.headers['user-agent'], "System error"]
                );
            } catch (auditError) {
                console.error("Failed to log audit:", auditError);
            }
        }
        res.status(500).json({
            errorMessage: "An error occurred during login.",
        });
    }
});

// Change Password Route
router.post("/change-password", auth, async (req, res) => {
    try {
        const { currentPassword, newPassword } = req.body;
        const userId = req.user;
        const ipAddress = getClientIP(req);
        const userAgent = req.headers['user-agent'];

        // Validation
        if (!currentPassword || !newPassword) {
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                [userId, ipAddress, userAgent, "Password change failed - Missing required fields"]
            );
            return res.status(400).json({
                errorMessage: "Please enter all required fields.",
            });
        }

        if (newPassword.length < 6) {
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                [userId, ipAddress, userAgent, "Password change failed - Password too short"]
            );
            return res.status(400).json({
                errorMessage: "New password must be at least 6 characters long.",
            });
        }

        // Get user
        const user = await vtdbquery("SELECT * FROM userstable WHERE id = ?", [userId]);
        if (user.length === 0) {
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                [userId, ipAddress, userAgent, "Password change failed - User not found"]
            );
            return res.status(404).json({
                errorMessage: "User not found.",
            });
        }

        // Verify current password
        const correctPassword = await bcrypt.compare(
            currentPassword,
            user[0].password
        );

        if (!correctPassword) {
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                [userId, ipAddress, userAgent, "Password change failed - Incorrect current password"]
            );
            return res.status(401).json({
                errorMessage: "Current password is incorrect.",
            });
        }

        // Start transaction
        await vtdbquery("START TRANSACTION");

        try {
            // Hash the new password
            const salt = await bcrypt.genSalt(10);
            const hashedPassword = await bcrypt.hash(newPassword, salt);

            // Update password
            await vtdbquery("UPDATE userstable SET password = ? WHERE id = ?", [hashedPassword, userId]);

            // Log successful password change
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'success', ?)",
                [userId, ipAddress, userAgent, "Password changed successfully"]
            );

            // Commit transaction
            await vtdbquery("COMMIT");

            res.json({
                success: true,
                message: "Password changed successfully.",
            });
        } catch (err) {
            // Rollback on error
            await vtdbquery("ROLLBACK");
            throw err;
        }
    } catch (err) {
        console.error("Change password error:", err);
        await vtdbquery(
            "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
            [userId, getClientIP(req), req.headers['user-agent'], "Password change failed - System error"]
        );
        res.status(500).json({
            errorMessage: "An error occurred while changing password.",
        });
    }
});

// User status route
router.get("/loggedIn", (req, res) => {
    try {
        const token = req.cookies.token || req.header('Authorization')?.replace('Bearer ', '');
        if (!token) return res.json(null);

        const validatedUser = jwt.verify(token, process.env.JWT_SECRET);

        res.json({ id: validatedUser.id, email: validatedUser.email, name: validatedUser.name, mobile: validatedUser.mobile, roles: validatedUser.roles, pbx_extension: validatedUser.pbx_extension });
    } catch (err) {
        return res.json(null);
    }
});

// User logout route
router.get("/logOut", async (req, res) => {
    try {
        // Clear the token cookie with proper options
        res.clearCookie('token', {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'strict',
            path: '/'
        });

        // Clear any other auth-related cookies
        res.clearCookie('user', {
            path: '/'
        });

        // Log the logout action
        if (req.user) {
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'success', ?)",
                [req.user, getClientIP(req), req.headers['user-agent'], "User logged out"]
            );
        }

        res.json({ success: true, message: 'Logged out successfully' });
    } catch (err) {
        console.error("Logout error:", err);
        res.status(500).json({
            errorMessage: "An error occurred during logout.",
        });
    }
});

//get user by id only for internal use need to verify the ip address
router.get("/internalusers/:id", async (req, res) => {
    const ip = req.headers['x-forwarded-for']?.split(',')[0] || req.socket?.remoteAddress;

    console.log(`Your IP is: ${ip}`);
    try {
        const { id } = req.params;
        const allowedIPS = ['49.206.197.32', '183.82.107.231', '122.175.11.35'];
        if (!allowedIPS.includes(ip)) {
            return res.status(403).json({
                errorMessage: "Unauthorized access.",
            });
        }
        const user = await vtdbquery("SELECT userstable.*,up.mobile,up.full_name FROM userstable LEFT JOIN user_profiles up ON userstable.id = up.user_id where userstable.id=?", [id]);
        res.json(user);
    } catch (err) {
        console.error("Get user error:", err);
        res.status(500).json({
            errorMessage: "An error occurred while fetching user.",
        });
    }
});
// Get all users (admin only)
router.get("/users", auth, checkRole(['Admin']), async (req, res) => {
    try {
        // Extract pagination parameters
        const offset = parseInt(req.query.offset) || 0; // 0-based page index
        const limit = parseInt(req.query.limit) || 10; // Default 10 users per page

        // Get total count of users
        const [countResult] = await vtdbquery(`
            SELECT COUNT(*) as total
            FROM userstable
        `);
        const total = parseInt(countResult.total);
        const totalPages = Math.ceil(total / limit);

        // Get paginated users with roles and branches aggregated
        const users = await vtdbquery(`
             SELECT 
              u.id AS user_id,
              u.userName,
              up.mobile,
              u.email,
              u.pbx_extension,
              u.status,
              COALESCE(ur.roles, JSON_ARRAY()) AS roles,
              COALESCE(ub.branches, JSON_ARRAY()) AS branches
            FROM (
              SELECT * 
              FROM userstable 
              ORDER BY id 
               LIMIT ? OFFSET ?
            ) u
            LEFT JOIN user_profiles up ON u.id = up.user_id
            LEFT JOIN (
              SELECT 
                ur.user_id,
                JSON_ARRAYAGG(JSON_OBJECT('role_id', r.id, 'role_name', r.name)) AS roles
              FROM user_roles ur
              JOIN roles r ON ur.role_id = r.id
              GROUP BY ur.user_id
            ) ur ON u.id = ur.user_id
            LEFT JOIN (
              SELECT 
                ub.user_id,
                JSON_ARRAYAGG(JSON_OBJECT('id', b.id, 'branchname', b.branchname,'can_assign_leads',ub.assign_leads)) AS branches
              FROM user_branches ub
              JOIN branches b ON ub.branch_id = b.id
              GROUP BY ub.user_id
            ) ub ON u.id = ub.user_id
            ORDER BY u.id
        `, [limit, offset]);

        // Parse JSON fields for roles and branches
        users.forEach(user => {
            user.roles = typeof user.roles === 'string' ? JSON.parse(user.roles) : user.roles;
            user.branches = typeof user.branches === 'string' ? JSON.parse(user.branches) : user.branches;
        });

        // Return paginated response
        res.json({
            items: users,
            total,
            offset,
            limit,
            totalPages
        });
    } catch (err) {
        console.error("Get users error:", err);
        res.status(500).json({
            errorMessage: "An error occurred while fetching users.",
        });
    }
});

// Update user status (admin only)
router.put("/users/:id/status", auth, checkRole(['Admin']), async (req, res) => {
    try {
        const { id } = req.params;
        const { status } = req.body;

        if (!['active', 'inactive', 'pending'].includes(status)) {
            return res.status(400).json({
                errorMessage: "Invalid status value.",
            });
        }

        await vtdbquery("UPDATE userstable SET status = ? WHERE id = ?", [status, id]);

        res.json({
            success: true,
            message: "User status updated successfully.",
        });
    } catch (err) {
        console.error("Update user status error:", err);
        res.status(500).json({
            errorMessage: "An error occurred while updating user status.",
        });
    }
});

// Update user roles (admin only)
router.put("/users/:id/roles", auth, checkRole(['Admin']), async (req, res) => {
    try {
        const { id } = req.params;
        const { roleIds } = req.body;
        console.log(roleIds);

        if (!Array.isArray(roleIds)) {
            return res.status(400).json({
                errorMessage: "roleIds must be an array of role IDs.",
            });
        }

        // Check if user exists
        const user = await vtdbquery("SELECT * FROM userstable WHERE id = ?", [id]);
        if (user.length === 0) {
            return res.status(404).json({
                errorMessage: "User not found.",
            });
        }

        // Check if all roles exist
        if (roleIds.length > 0) {
            const roleCount = await vtdbquery(
                "SELECT COUNT(*) as count FROM roles WHERE id IN (?)",
                [roleIds]
            );
            console.log(roleCount);
            if (parseInt(roleCount[0].count) !== roleIds.length) {
                return res.status(400).json({
                    errorMessage: "One or more roles do not exist.",
                });
            }
        } else {
            return res.status(400).json({
                errorMessage: "User must have at least one role.",
            });
        }

        // Start a transaction
        await vtdbquery("START TRANSACTION");
        try {
            // Remove existing roles
            await vtdbquery("DELETE FROM user_roles WHERE user_id = ?", [id]);

            // Add new roles
            if (roleIds.length > 0) {
                // Build a proper SQL query with multiple VALUES clauses
                const placeholders = roleIds.map(() => "(?, ?)").join(", ");
                const params = [];

                // Flatten the parameters: [id, roleId1, id, roleId2, ...]
                roleIds.forEach(roleId => {
                    params.push(id);
                    params.push(roleId);
                });

                await vtdbquery(
                    `INSERT INTO user_roles (user_id, role_id) VALUES ${placeholders}`,
                    params
                );
            }

            // Commit the transaction
            await vtdbquery("COMMIT");

            // Get updated roles for response
            const roles = await vtdbquery(`
                SELECT r.id, r.name 
                FROM user_roles ur
                JOIN roles r ON ur.role_id = r.id
                WHERE ur.user_id = ?
            `, [id]);

            res.json({
                success: true,
                message: "User roles updated successfully.",
                roles: roles
            });
        } catch (err) {
            // Rollback on error
            await vtdbquery("ROLLBACK");
            throw err;
        }
    } catch (err) {
        console.error("Update user roles error:", err);
        res.status(500).json({
            errorMessage: "An error occurred while updating user roles.",
        });
    }
});

// Get a specific user by ID (admin only)
router.get("/users/:id", auth, checkRole(['Admin']), async (req, res) => {
    try {
        const { id } = req.params;
        // Get user data
        const users = await vtdbquery(`
            SELECT u.id, u.email, u.status, u.last_login, u.created_at, 
                   up.full_name, up.mobile, up.address, up.profile_image
            FROM userstable u
            LEFT JOIN user_profiles up ON u.id = up.user_id
            WHERE u.id = ?
        `, [id]);
        if (users.length === 0) {
            return res.status(404).json({
                errorMessage: "User not found.",
            });
        }
        console.log("users", users);
        const user = users[0];
        // Get user roles
        const roles = await vtdbquery(`
            SELECT r.id, r.name 
            FROM user_roles ur
            JOIN roles r ON ur.role_id = r.id
            WHERE ur.user_id = ?
        `, [id]);
        user.roles = roles;
        // Get user branches
        const branches = await vtdbquery(`
            SELECT b.id, b.branchname 
            FROM user_branches ub
            JOIN branches b ON ub.branch_id = b.id
            WHERE ub.user_id = ?
        `, [id]);
        user.branches = branches;
        res.json(user);
    } catch (err) {
        console.error("Get user error:", err);
        res.status(500).json({
            errorMessage: "An error occurred while fetching user.",
        });
    }
});

// create a user (admin only)
router.post("/users", auth, checkRole(['Admin']), async (req, res) => {

    try {
        const { email, userName, mobile = null, pbx_extension = null, role_ids, status, branch_ids } = req.body;

        // Validation
        if (!email || !userName) {
            return res.status(400).json({
                errorMessage: "Please enter all required fields.",
            });
        }

        // Check if user already exists
        const existingUser = await vtdbquery("SELECT * FROM userstable WHERE email = ?", [email]);
        if (existingUser.length > 0) {
            return res.status(400).json({
                errorMessage: "An account with this email already exists.",
            });
        }

        // Create the user
        const result = await vtdbquery(
            "INSERT INTO userstable (email, userName, password, status, pbx_extension) VALUES (?, ?, ?, ?, ?)",
            [email, userName, 'test', status || 'pending', pbx_extension || null]
        );

        const userId = result.insertId;
        // Update role_ids if provided
        if (role_ids) {
            if (role_ids.length > 0) {
                const values = role_ids.map(() => "(?, ?)").join(", ");
                const params = role_ids.reduce((acc, roleId) => [...acc, userId, roleId], []);
                await vtdbquery(`INSERT INTO user_roles (user_id, role_id) VALUES ${values}`, params);
            }
        }
        // Update branch_ids if provided
        if (branch_ids && branch_ids.length > 0) {
            const values = branch_ids.map(() => "(?, ?)").join(", ");
            const params = branch_ids.reduce((acc, branchId) => [...acc, userId, branchId], []);
            await vtdbquery(`INSERT INTO user_branches (user_id, branch_id) VALUES ${values}`, params);
        }
        if (mobile !== null) {
            await vtdbquery("INSERT INTO user_profiles (user_id, full_name, mobile) VALUES (?, ?, ?)", [userId, userName, mobile]);
        }

        res.status(201).json({
            success: true,
            message: "User created successfully.",
        });
    } catch (err) {
        console.error("Registration error:", err);
        res.status(500).json({
            errorMessage: "An error occurred during registration.",
        });
    }
});

// Update a user (admin only)
router.put("/users/:id", auth, checkRole(['Admin']), async (req, res) => {
    try {
        const { id } = req.params;
        const { email, mobile = null, address = null, status, role_ids, userName, branch_ids, pbx_extension = null, can_assign_leads = null } = req.body;

        // Check if user exists
        const existingUser = await vtdbquery("SELECT * FROM userstable WHERE id = ?", [id]);
        if (existingUser.length === 0) {
            return res.status(404).json({
                errorMessage: "User not found.",
            });
        }

        // Check if email is already taken by another user
        if (email && email !== existingUser[0].email) {
            const emailCheck = await vtdbquery("SELECT * FROM userstable WHERE email = ? AND id != ?", [email, id]);
            if (emailCheck.length > 0) {
                return res.status(400).json({
                    errorMessage: "Email is already in use by another account.",
                });
            }
            await vtdbquery("UPDATE userstable SET email = ? WHERE id = ?", [email, id]);
        }
        if (status && ['active', 'inactive', 'pending'].includes(status)) {
            await vtdbquery("UPDATE userstable SET status = ? WHERE id = ?", [status, id]);
        }
        if (userName && userName !== existingUser[0].userName) {
            await vtdbquery("UPDATE userstable SET userName = ? WHERE id = ?", [userName, id]);
        }
        if (pbx_extension !== null) {
            await vtdbquery("UPDATE userstable SET pbx_extension = ? WHERE id = ?", [pbx_extension, id]);
        }
        if (role_ids) {
            const existingRoles = await vtdbquery("SELECT * FROM user_roles WHERE user_id = ?", [id]);
            const rolesToAdd = role_ids.filter(role => !existingRoles.some(existingRole => existingRole.role_id === role));
            const rolesToRemove = existingRoles.filter(existingRole => !role_ids.includes(existingRole.role_id));
            if (rolesToAdd.length > 0) {
                const values = rolesToAdd.map(() => "(?, ?)").join(", ");
                const params = rolesToAdd.reduce((acc, roleId) => [...acc, id, roleId], []);
                await vtdbquery(`INSERT INTO user_roles (user_id, role_id) VALUES ${values}`, params);
            }
            if (rolesToRemove.length > 0) {
                await vtdbquery(
                    "DELETE FROM user_roles WHERE user_id = ? AND role_id IN (?)",
                    [id, rolesToRemove.map(role => role.role_id)]
                );
            }
        }
        // Update branch_ids if provided
        if (branch_ids) {
            const existingBranches = await vtdbquery("SELECT * FROM user_branches WHERE user_id = ?", [id]);
            const branchesToAdd = branch_ids.filter(branch => !existingBranches.some(existingBranch => existingBranch.branch_id === branch));
            const branchesToRemove = existingBranches.filter(existingBranch => !branch_ids.includes(existingBranch.branch_id));
            if (branchesToAdd.length > 0) {
                const values = branchesToAdd.map(() => "(?, ?, ?)").join(", ");
                const params = branchesToAdd.reduce((acc, branchId) => [...acc, id, branchId, can_assign_leads], []);
                await vtdbquery(`INSERT INTO user_branches (user_id, branch_id, assign_leads) VALUES ${values}`, params);
            }
            if (branchesToRemove.length > 0) {
                await vtdbquery(
                    "DELETE FROM user_branches WHERE user_id = ? AND branch_id IN (?)",
                    [id, branchesToRemove.map(branch => branch.branch_id)]
                );
            }
        }
        if (can_assign_leads !== null) {
            await vtdbquery("UPDATE user_branches SET assign_leads = ? WHERE user_id = ?", [can_assign_leads, id]);
        }

        // Update profile
        const profiles = await vtdbquery("SELECT * FROM user_profiles WHERE user_id = ?", [id]);
        if (profiles.length === 0) {
            await vtdbquery(
                "INSERT INTO user_profiles (user_id, full_name, mobile, address) VALUES (?, ?, ?, ?)",
                [id, userName, mobile || null, address || null]
            );
        } else {
            await vtdbquery(
                "UPDATE user_profiles SET full_name = ?, mobile = ?, address = ? WHERE user_id = ?",
                [
                    userName,
                    mobile !== undefined ? mobile : profiles[0].mobile,
                    address !== undefined ? address : profiles[0].address,
                    id
                ]
            );
        }
        res.json({
            success: true,
            message: "User updated successfully.",
        });
    } catch (err) {
        console.error("Update user error:", err);
        res.status(500).json({
            errorMessage: "An error occurred while updating user.",
        });
    }
});

// Reset user password (admin only)
router.post("/users/:id/reset-password", auth, checkRole(['Admin']), async (req, res) => {
    try {
        const { id } = req.params;
        const { newPassword } = req.body;

        // Validation
        if (!newPassword || newPassword.length < 6) {
            return res.status(400).json({
                errorMessage: "New password must be at least 6 characters long.",
            });
        }

        // Check if user exists
        const existingUser = await vtdbquery("SELECT * FROM userstable WHERE id = ?", [id]);
        if (existingUser.length === 0) {
            return res.status(404).json({
                errorMessage: "User not found.",
            });
        }

        // Hash the new password
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(newPassword, salt);

        // Update the password
        await vtdbquery("UPDATE userstable SET password = ? WHERE id = ?", [hashedPassword, id]);

        res.json({
            success: true,
            message: "User password reset successfully.",
        });
    } catch (err) {
        console.error("Reset password error:", err);
        res.status(500).json({
            errorMessage: "An error occurred while resetting the password.",
        });
    }
});

// Get login audit logs (admin only)
router.get("/login-audit", auth, checkRole(['Admin']), async (req, res) => {
    try {
        // Extract pagination and filter parameters
        const page = parseInt(req.query.page) || 0;
        const limit = parseInt(req.query.limit) || 10;
        const offset = page * limit;
        const userId = req.query.userId;
        const status = req.query.status;
        const startDate = req.query.startDate;
        const endDate = req.query.endDate;

        // Build the WHERE clause based on filters
        let whereClause = "WHERE 1=1";
        const params = [];

        if (userId) {
            whereClause += " AND la.user_id = ?";
            params.push(userId);
        }
        if (status) {
            whereClause += " AND la.status = ?";
            params.push(status);
        }
        if (startDate) {
            whereClause += " AND DATE(la.login_time) >= ?";
            params.push(startDate);
        }
        if (endDate) {
            whereClause += " AND DATE(la.login_time) <= ?";
            params.push(endDate);
        }

        // Get total count
        const [countResult] = await vtdbquery(`
            SELECT COUNT(*) as total
            FROM login_audit la
            ${whereClause}
        `, params);

        const total = parseInt(countResult.total);
        const totalPages = Math.ceil(total / limit);

        // Get paginated audit logs
        const auditLogs = await vtdbquery(`
            SELECT 
                la.*,
                u.email,
                up.full_name
            FROM login_audit la
            LEFT JOIN userstable u ON la.user_id = u.id
            LEFT JOIN user_profiles up ON u.id = up.user_id
            ${whereClause}
            ORDER BY la.login_time DESC
            LIMIT ? OFFSET ?
        `, [...params, limit, offset]);

        res.json({
            items: auditLogs,
            total,
            page,
            limit,
            totalPages
        });
    } catch (err) {
        console.error("Get login audit logs error:", err);
        res.status(500).json({
            errorMessage: "An error occurred while fetching login audit logs.",
        });
    }
});

// Forgot Password - Request OTP
router.post("/forgot-password/request", async (req, res) => {
    try {
        const { email } = req.body;

        // Validate email domain
        if (!email.endsWith('@vectorindia.org') && !email.endsWith('@vectorinstitute.in')) {
            return res.status(400).json({
                errorMessage: "Please use your Vector India email address (@vectorindia.org) or Vector Institute email address (@vectorinstitute.in)",
            });
        }

        // Check if user exists
        const user = await vtdbquery("SELECT * FROM userstable WHERE email = ?", [email]);
        if (user.length === 0) {
            return res.status(404).json({
                errorMessage: "No account found with this email address.",
            });
        }

        // Generate and store OTP
        const otp = generateOTP();
        otpStore.set(email, {
            code: otp,
            timestamp: Date.now(),
            attempts: 0
        });

        // Send email
        const mailOptions = {
            from: {
                name: "Vector India",
                address: process.env.SMTP_USER
            },
            to: email,
            subject: 'Password Reset OTP - Vector India',
            html: `
                <h2>Password Reset Request</h2>
                <p>Your OTP for password reset is: <strong>${otp}</strong></p>
                <p>This OTP will expire in 10 minutes.</p>
                <p>If you didn't request this, please ignore this email.</p>
            `
        };

        const info = await transporter.sendMail(mailOptions);
        console.log("Message sent: %s", info.messageId);

        res.json({
            success: true,
            message: "OTP has been sent to your email address.",
        });
    } catch (err) {
        console.error("Forgot password error:", err);
        res.status(500).json({
            errorMessage: "Failed to process password reset request.",
        });
    }
});

// Verify OTP and Reset Password
router.post("/forgot-password/reset", async (req, res) => {
    try {
        const { email, otp, newPassword } = req.body;
        const ipAddress = getClientIP(req);
        const userAgent = req.headers['user-agent'];

        // Validate password
        if (!newPassword || newPassword.length < 6) {
            return res.status(400).json({
                errorMessage: "New password must be at least 6 characters long.",
            });
        }

        // Check OTP
        const otpData = otpStore.get(email);
        if (!otpData) {
            return res.status(400).json({
                errorMessage: "OTP has expired. Please request a new one.",
            });
        }

        // Get user details for audit logging
        const user = await vtdbquery("SELECT id FROM userstable WHERE email = ?", [email]);
        if (user.length === 0) {
            return res.status(404).json({
                errorMessage: "User not found.",
            });
        }
        const userId = user[0].id;

        // Validate OTP expiry (10 minutes)
        if (Date.now() - otpData.timestamp > 10 * 60 * 1000) {
            otpStore.delete(email);
            // Log failed attempt due to expired OTP
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                [userId, ipAddress, userAgent, "Password reset failed - OTP expired"]
            );
            return res.status(400).json({
                errorMessage: "OTP has expired. Please request a new one.",
            });
        }

        // Check attempts
        if (otpData.attempts >= 3) {
            otpStore.delete(email);
            // Log failed attempt due to too many attempts
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                [userId, ipAddress, userAgent, "Password reset failed - Too many invalid attempts"]
            );
            return res.status(400).json({
                errorMessage: "Too many invalid attempts. Please request a new OTP.",
            });
        }

        // Verify OTP
        if (otpData.code !== otp) {
            otpData.attempts++;
            // Log failed attempt due to invalid OTP
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                [userId, ipAddress, userAgent, "Password reset failed - Invalid OTP"]
            );
            return res.status(400).json({
                errorMessage: "Invalid OTP.",
                attemptsLeft: 3 - otpData.attempts
            });
        }

        // Start transaction for password update and audit logging
        await vtdbquery("START TRANSACTION");

        try {
            // Hash new password
            const salt = await bcrypt.genSalt(10);
            const hashedPassword = await bcrypt.hash(newPassword, salt);

            // Update password
            await vtdbquery("UPDATE userstable SET password = ? WHERE email = ?", [hashedPassword, email]);

            // Log successful password reset
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'success', ?)",
                [userId, ipAddress, userAgent, "Password reset successful via forgot password"]
            );

            // Commit transaction
            await vtdbquery("COMMIT");

            // Clear OTP
            otpStore.delete(email);

            res.json({
                success: true,
                message: "Password has been reset successfully.",
            });
        } catch (err) {
            // Rollback on error
            await vtdbquery("ROLLBACK");
            throw err;
        }
    } catch (err) {
        console.error("Password reset error:", err);
        // If we have the userId, log the error
        if (userId) {
            await vtdbquery(
                "INSERT INTO login_audit (user_id, login_time, ip_address, user_agent, status, failure_reason) VALUES (?, NOW(), ?, ?, 'failed', ?)",
                [userId, getClientIP(req), req.headers['user-agent'], "Password reset failed - System error"]
            );
        }
        res.status(500).json({
            errorMessage: "Failed to reset password.",
        });
    }
});

module.exports = router;
