const jwt = require("jsonwebtoken");
const { query } = require('../config/db');

// Basic authentication middleware
function auth(req, res, next) {
    try {
        const token = req.header('Authorization')?.replace('Bearer ', '');

        if (!token) return res.status(401).json({ errorMessage: "Unauthorized." });

        const validatedUser = jwt.verify(token, process.env.JWT_SECRET);
        req.user = validatedUser.id;
        req.email = validatedUser.email;
        req.name = validatedUser.name;
        req.userMobile = validatedUser.mobile;
        req.roles = validatedUser.roles;
        req.pbx_extension = validatedUser.pbx_extension;
        next();
    } catch (err) {
        return res.status(401).json({ errorMessage: "Unauthorized." });
    }
}

// Role-based access control middleware
function checkRole(roles = []) {
    return function (req, res, next) {
        if (!req.user) {
            return res.status(401).json({ errorMessage: "Unauthorized." });
        }

        // Get user roles
        query(`
            SELECT r.name 
            FROM user_roles ur
            JOIN roles r ON ur.role_id = r.id
            WHERE ur.user_id = ?
        `, [req.user])
            .then(userRoles => {
                const userRoleNames = userRoles.map(role => role.name);

                // Check if user has any of the required roles
                const hasRequiredRole = roles.length === 0 || roles.some(role => userRoleNames.includes(role));

                if (!hasRequiredRole) {
                    return res.status(403).json({ errorMessage: "Access denied. Insufficient permissions." });
                }

                next();
            })
            .catch(err => {
                console.error("Role check error:", err);
                return res.status(500).json({ errorMessage: "Internal server error." });
            });
    };
}

module.exports = { auth, checkRole }; 