const express = require("express");
const cors = require("cors");
const cookieParser = require("cookie-parser");
const path = require("path");
const fs = require("fs");
const dotenv = require("dotenv");
process.env.DOTENV_CONFIG_QUIET = true;
dotenv.config({ silent: true })
const { getFormattedDateTime } = require('./routes/utils/dateHelper');

// Create express app
const app = express();
const PORT = process.env.PORT || 5071;
const GLOBAL_CONFIG = {
    apiBaseUrl: process.env.API_BASE_URL || "https://vectorindia.org/api",
    BaseUrl: process.env.BASE_URL || "https://vectorindia.org",
    env: process.env.NODE_ENV || "DEV",
    buildTime: getFormattedDateTime(),
};

app.use((req, res, next) => {
    // If the URL starts with //, replace it with /
    if (req.url.startsWith('//')) {
        req.url = req.url.replace(/^\/\//, '/');
    }
    // //console.log("req.url: ", req.url);
    next();
});

// Middleware for all requests
// Request logging middleware
app.use((req, res, next) => {
    console.log(`${getFormattedDateTime()} - ${req.method} ${req.path}`);
    next();
});

app.use(express.json({
    limit: '50mb',
    verify: (req, res, buf) => {
        // Store raw body for PhonePe webhook verification
        req.rawBody = buf;
    }
}));

// Parse URL-encoded form data
app.use(express.urlencoded({
    extended: true,
    limit: '50mb'
}));

app.use((err, req, res, next) => {
    if (err.type === 'entity.too.large') {
        res.status(413).send('Payload Too Large');
    } else {
        next(err);
    }
});


const allowed = new Set([
    "http://localhost:3000",
    "http://localhost:5071",
    "https://vectorindia.org",
    "https://wwwvector.vectorindia.org",
    "https://www.vectorindia.org"
]);
app.use(cors({
    origin: function (origin, callback) {
        // allow requests with no origin (mobile apps, curl, server-to-server)
        if (!origin) return callback(null, true);

        if (allowed.has(origin)) {
            return callback(null, true);
        } else {
            console.warn('Blocked CORS origin:', origin);
            return callback(new Error('Not allowed by CORS'), false);
        }
    },
    credentials: true,
}));

app.use(cookieParser());



// Custom middleware to handle BigInt in JSON
app.use((req, res, next) => {
    express.json({
        reviver: (key, value) => {
            if (typeof value === 'string' && /^\d+n$/.test(value)) {
                return BigInt(value.slice(0, -1)); // Convert string with 'n' back to BigInt
            }
            return value;
        }
    })(req, res, (err) => {
        if (err) {
            return res.status(400).send({ error: 'Invalid JSON' });
        }
        next();
    });
});

// Custom JSON.stringify function to handle BigInt
app.use((req, res, next) => {
    res.json = function (data) {
        return res.send(
            JSON.stringify(data, (key, value) => {
                if (typeof value === 'bigint') {
                    return value.toString() + 'n'; // Convert BigInt to string with 'n' suffix
                }
                return value;
            })
        );
    };
    next();
});


app.get("/preconfig.js", (req, res) => {
    res.type("application/javascript");
    res.send(`window.AppConfig = ${JSON.stringify(GLOBAL_CONFIG)};`);
});

app.get("/registration", (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'register.html'));
});

// Serve static files from the uploads directory
app.use(express.static(path.join(__dirname, 'build')));
app.use(express.static(path.join(__dirname, 'public')));

// Import routes

const collegeRouter = require('./routes/college');
const registrationRouter = require('./routes/registration/register');
const { router: verifiotpRouter } = require('./routes/verifiotp');
// API routes
app.use("/api/college", collegeRouter);
app.use("/api/registration", registrationRouter);
app.use("/api/verification", verifiotpRouter);


// Health check route
app.get("/api/health", (req, res) => {
    res.json({ status: "ok", environment: process.env.NODE_ENV, date: getFormattedDateTime() });
});

// Error handling middleware for route parsing errors
app.use((err, req, res, next) => {
    if (err instanceof TypeError && err.message.includes('Missing parameter name')) {
        console.error('Route parsing error:', err.message);
        console.error('URL that caused error:', req.url);
        return res.status(400).json({ error: 'Invalid route pattern' });
    }
    next(err);
});



// Catch-all route for client-side routing - Express 5.x compatible (must be registered AFTER API routes)
app.get('/*splat', function (req, res) {
    // Check if the requested path exists as a static file
    const requestedPath = path.join(__dirname, 'public', req.path);

    // If it's a request for an HTML file that doesn't exist, serve 404
    if (req.path.endsWith('.html') || req.path.match(/^\/[^.]*$/)) {
        // Check if the file exists in public directory
        if (req.path.endsWith('.html')) {
            const filePath = path.join(__dirname, 'public', req.path);
            if (!fs.existsSync(filePath)) {
                return res.status(404).sendFile(path.join(__dirname, 'public', '404.html'));
            }
        } else {
            // For routes without extension, check if HTML file exists
            const htmlPath = path.join(__dirname, 'public', req.path + '.html');
            if (fs.existsSync(htmlPath)) {
                return res.sendFile(htmlPath);
            } else {
                // Check if it's a directory with index.html
                const indexPath = path.join(__dirname, 'public', req.path, 'index.html');
                if (fs.existsSync(indexPath)) {
                    return res.sendFile(indexPath);
                } else {
                    // Serve 404 page
                    return res.status(404).sendFile(path.join(__dirname, 'public', '404.html'));
                }
            }
        }
    }

    // For all other routes, serve the index.html file
    // This allows React Router to handle client-side routing
    res.sendFile(path.join(__dirname, 'build', 'index.html'));
});

// 404 handler for static files
app.use((req, res, next) => {
    // Check if the request is for a static file that doesn't exist
    if (req.path.match(/\.(html|css|js|png|jpg|jpeg|gif|svg|ico|pdf|woff|woff2|ttf|eot)$/)) {
        return res.status(404).sendFile(path.join(__dirname, 'public', '404.html'));
    }
    next();
});
// Start HTTPS server if SSL certificates are available
if (app) {
    app.listen(PORT, () => {
        console.log(`Environment: ${process.env.NODE_ENV}`);
        console.log(`HTTP Server running on port ${PORT}`);
    });
} else {
    console.log('HTTP server not started due to missing SSL certificates');
}