const express = require('express');
const cors = require('cors');
const cookieParser = require('cookie-parser');
const path = require('path');
const fs = require('fs');

const configureExpress = (app) => {
    // Enable CORS
    // CORS configuration
    app.use(
        cors({
            origin: ["http://localhost:3000", "https://wwwvector.vectorindia.org", "https://vectorindia.org","https://www.vectorindia.org/"],
            credentials: true,
        })
    );

    // Parse JSON bodies
    app.use(express.json());

    // Parse URL-encoded bodies
    app.use(express.urlencoded({ extended: true }));

    // Add security headers
    app.use((req, res, next) => {
        res.setHeader('X-Content-Type-Options', 'nosniff');
        res.setHeader('X-Frame-Options', 'DENY');
        res.setHeader('X-XSS-Protection', '1; mode=block');
        next();
    });

    // URL normalization middleware
    app.use((req, res, next) => {
        if (req.url.startsWith('//')) {
            req.url = req.url.replace(/^\/\//, '/');
        }
        console.log("req.url : ", req.url);
        next();
    });

    // Cookie parser
    app.use(cookieParser());

    // Payload size error handler
    app.use((err, req, res, next) => {
        if (err.type === 'entity.too.large') {
            res.status(413).send('Payload Too Large');
        } else {
            next(err);
        }
    });

    // BigInt JSON handler middleware
    app.use((req, res, next) => {
        express.json({
            reviver: (key, value) => {
                if (typeof value === 'string' && /^\d+n$/.test(value)) {
                    return BigInt(value.slice(0, -1));
                }
                return value;
            }
        })(req, res, (err) => {
            if (err) {
                return res.status(400).send({ error: 'Invalid JSON' });
            }
            next();
        });
    });

    // Custom JSON stringify for BigInt
    app.use((req, res, next) => {
        res.json = function (data) {
            return res.send(
                JSON.stringify(data, (key, value) => {
                    if (typeof value === 'bigint') {
                        return value.toString() + 'n';
                    }
                    return value;
                })
            );
        };
        next();
    });
};

// post configuration
const postConfig = (app) => {
    app.use(express.static(path.join(__dirname, 'build')));

    app.get('/*', function (req, res) {
        console.log("Original URL:", req.url);

        // Remove any double slashes
        let cleanUrl = req.url.replace(/\/+/g, '/');

        let urls = cleanUrl.split('/build/');
        var item = 'index.html';

        if (urls.length === 1) {
            return res.status(404).send("Resource not found");
        }

        if (urls[1] !== '') {
            item = urls[1];
        }

        // Check if file exists and serve appropriately
        fs.access(path.join(__dirname, 'build', item), fs.constants.F_OK, (err) => {
            if (err) {
                console.log("Serving index.html for:", path.join(__dirname, 'build', item));
                res.sendFile(path.join(__dirname, 'build', 'index.html'));
            } else {
                console.log("Serving file:", path.join(__dirname, 'build', item));
                res.sendFile(path.join(__dirname, 'build', item));
            }
        });
    });

    // Handle 404 errors
    app.use((req, res) => {
        res.status(404).json({ message: 'Route not found' });
    });

    // Error handling middleware
    app.use((err, req, res, next) => {
        console.error(err.stack);
        res.status(500).json({
            message: 'Something went wrong!',
            error: process.env.NODE_ENV === 'development' ? err.message : undefined
        });
    });
};

module.exports = { configureExpress, postConfig };
