/**
 * Custom College Dropdown with Infinite Scroll and Search
 * Designed for Vector India Registration Form
 */

(function ($) {
    'use strict';

    // Initialize custom college dropdown
    function initCustomCollegeDropdown() {
        var currentPage = 1;
        var isLoading = false;
        var hasMore = true;
        var searchTimeout = null;
        var currentStateId = null;
        var selectedCollege = null;

        // Function to create college item HTML
        function createCollegeItemHTML(college) {
            var html = '<div class="college-dropdown-item" data-college-id="' + college.id + '" data-college-name="' + college.college_name + '">';
            html += '<span class="college-name">' + college.college_name + '</span>';

            var details = [];
            if (college.univerisity_name) {
                details.push('<span class="college-detail-item">🎓 ' + college.univerisity_name + '</span>');
            }
            if (college.college_address) {
                details.push('<span class="college-detail-item">📍 ' + college.college_address + '</span>');
            }
            if (college.college_website) {
                details.push('<span class="college-detail-item">🌐 ' + college.college_website + '</span>');
            }

            if (details.length > 0) {
                html += '<div class="college-details">' + details.join('') + '</div>';
            }

            html += '</div>';
            return html;
        }

        // Function to load colleges
        function loadColleges(stateId, page, search, append) {
            if (isLoading || (!append && !hasMore && page > 1)) return;
            isLoading = true;

            // Add loading indicator
            if (!append) {
                $("#college_dropdown_list").html('<div class="college-dropdown-item loading"><span class="loading-spinner"></span>Loading colleges...</div>');

                // Also update hidden select
                $("#college_name").empty().append('<option value="">Loading...</option>');
            } else {
                // Remove "other" option and add loading
                $("#college_dropdown_list .other-college").remove();
                $("#college_dropdown_list").append('<div class="college-dropdown-item loading"><span class="loading-spinner"></span>Loading more...</div>');
            }

            var url = (window.AppConfig ? window.AppConfig.apiBaseUrl : 'https://vectorindia.org/api') + '/college/colleges/' + stateId + '?page=' + page + '&limit=100';
            if (search && search.length >= 3) {
                url += '&search=' + encodeURIComponent(search);
            }

            $.ajax({
                type: 'GET',
                url: url,
                dataType: 'json',
                success: function (response) {
                    console.log('Colleges loaded:', response);
                    isLoading = false;

                    if (response.success && response.colleges && response.colleges.length > 0) {
                        // Remove loading indicator
                        $("#college_dropdown_list .loading").remove();

                        if (!append) {
                            // Keep the "Other College" option when clearing
                            $("#college_dropdown_list").find('.college-dropdown-item:not(.other-college)').remove();
                            $("#college_name").empty().append('<option value="">Select College</option>');
                        }

                        // Show college count
                        var totalColleges = response.pagination ? response.pagination.total : response.colleges.length;
                        var searchText = search ? ' for "' + search + '"' : '';
                        $("#college_count_text").text(totalColleges + ' colleges found' + searchText);
                        $("#college_count_display").show();

                        // Add colleges to custom dropdown
                        $.each(response.colleges, function (index, college) {
                            var collegeHTML = createCollegeItemHTML(college);
                            $("#college_dropdown_list").append(collegeHTML);

                            // Also update hidden select for form submission
                            if (!$("#college_name option[value='" + college.id + "']").length) {
                                $("<option/>")
                                    .text(college.college_name)
                                    .val(college.id)
                                    .data('college', college)
                                    .appendTo("#college_name");
                            }
                        });

                        // Ensure "Other College" option is at the end (only if not already there)
                        if (!$("#college_dropdown_list .other-college").length) {
                            $("#college_dropdown_list").append('<div class="college-dropdown-item other-college" data-college-id="other">✏️ Other College (Not in list)</div>');
                        }

                        // Add to hidden select
                        if (!$("#college_name option[value='other']").length) {
                            $("#college_name").append('<option value="other">Other College</option>');
                        }

                        // Update pagination state
                        if (response.pagination) {
                            hasMore = response.pagination.page < response.pagination.totalPages;
                            console.log('Pagination:', response.pagination, 'Has more:', hasMore);
                        } else {
                            hasMore = false;
                        }

                        // Bind click events to new items
                        bindCollegeItemEvents();
                    } else if (!append) {
                        // No colleges found - but still show "Other College" option
                        $("#college_dropdown_list").html('<div class="college-dropdown-item placeholder">No colleges found</div>');

                        // Add "Other College" option even when no colleges found
                        $("#college_dropdown_list").append('<div class="college-dropdown-item other-college" data-college-id="other">✏️ Other College (Not in list)</div>');

                        // Update hidden select
                        $("#college_name").empty().append('<option value="">No colleges found</option>');
                        $("#college_name").append('<option value="other">Other College</option>');

                        $("#college_count_display").hide();

                        // Bind click events to the "Other College" option
                        bindCollegeItemEvents();
                    }
                },
                error: function (xhr, status, error) {
                    console.error('Error loading colleges:', error);
                    isLoading = false;
                    $("#college_dropdown_list .loading").remove();
                    if (!append) {
                        $("#college_dropdown_list").html('<div class="college-dropdown-item placeholder">Error loading colleges. Please try again.</div>');

                        // Add "Other College" option even on error
                        $("#college_dropdown_list").append('<div class="college-dropdown-item other-college" data-college-id="other">✏️ Other College (Not in list)</div>');

                        // Update hidden select
                        $("#college_name").empty().append('<option value="">Error loading colleges</option>');
                        $("#college_name").append('<option value="other">Other College</option>');

                        $("#college_count_display").hide();

                        // Bind click events to the "Other College" option
                        bindCollegeItemEvents();
                    }
                }
            });
        }

        // Bind click events to college items
        function bindCollegeItemEvents() {
            $("#college_dropdown_list .college-dropdown-item:not(.placeholder):not(.loading)").off('click').on('click', function () {
                var collegeId = $(this).data('college-id');
                var collegeName = $(this).data('college-name') || 'Other College';

                // Remove previous selection
                $("#college_dropdown_list .college-dropdown-item").removeClass('selected');

                // Add selection to clicked item
                $(this).addClass('selected');

                // Update hidden select
                $("#college_name").val(collegeId);
                $("#college_name").trigger('change');

                // Update college_code and college_name_full fields
                if (collegeId === 'other') {
                    // Clear college_code and college_name_full when "Other College" is selected
                    $("#college_code").val('');
                    $("#college_name_full").val('');
                    $(".college-selected-text").text('Other College (Not in list)').removeClass('placeholder');
                    $("#other_colledge_name").show().focus();
                } else {
                    // Set college_code from the selected college data
                    var collegeData = $("#college_name option[value='" + collegeId + "']").data('college');
                    if (collegeData && collegeData.college_code) {
                        $("#college_code").val(collegeData.college_code);
                    } else {
                        $("#college_code").val(collegeId); // Fallback to college ID if no college_code
                    }

                    // Set college_name_full with the full college name
                    $("#college_name_full").val(collegeName);

                    $(".college-selected-text").html('<strong>' + collegeName + '</strong>').removeClass('placeholder');
                    $("#other_colledge_name").hide();
                }

                // Close dropdown
                $("#college_dropdown_header").removeClass('active');
                $("#college_dropdown_list").removeClass('active');
            });
        }

        // Toggle dropdown
        $(document).on('click', '#college_dropdown_header', function (e) {
            e.stopPropagation();
            $(this).toggleClass('active');
            $("#college_dropdown_list").toggleClass('active');
        });

        // Close dropdown when clicking outside
        $(document).on('click', function (e) {
            if (!$(e.target).closest('.custom-college-dropdown').length &&
                !$(e.target).closest('.college-search-input').length) {
                $("#college_dropdown_header").removeClass('active');
                $("#college_dropdown_list").removeClass('active');
            }
        });

        // Handle state change
        $("#college_state").on('change', function () {
            currentStateId = $(this).val();

            if (currentStateId) {
                currentPage = 1;
                hasMore = true;

                // Show custom dropdown
                $("#custom_college_dropdown_wrapper").show();

                // Reset dropdown
                $(".college-selected-text").text('Select College Name').addClass('placeholder');
                $("#college_search").val(''); // Clear search
                $("#college_name").val(''); // Clear hidden select
                $("#college_code").val(''); // Clear college code
                $("#college_name_full").val(''); // Clear college name full
                $("#other_colledge_name").hide(); // Hide other college input

                // Show "Other College" option immediately
                $("#college_dropdown_list .other-college").show();

                // Load colleges
                loadColleges(currentStateId, currentPage, '', false);

                // Auto-open dropdown after loading colleges
                setTimeout(function () {
                    $("#college_dropdown_header").addClass('active');
                    $("#college_dropdown_list").addClass('active');
                }, 500); // Wait for colleges to load
            } else {
                // Hide custom dropdown
                $("#custom_college_dropdown_wrapper").hide();
                $("#college_name").val('');
                $("#college_code").val('');
                $("#college_name_full").val('');
                $("#college_dropdown_list").empty();
                $("#college_count_display").hide();

                // Hide "Other College" option
                $("#college_dropdown_list .other-college").hide();
            }
        });

        // Handle infinite scroll on custom dropdown
        $("#college_dropdown_list").on('scroll', function () {
            var scrollTop = $(this).scrollTop();
            var scrollHeight = this.scrollHeight;
            var clientHeight = $(this).outerHeight();

            // Check if scrolled to bottom (with 50px threshold for better UX)
            if (scrollTop + clientHeight >= scrollHeight - 50 && hasMore && currentStateId && !isLoading) {
                console.log('Loading more colleges... Page:', currentPage + 1);
                currentPage++;
                loadColleges(currentStateId, currentPage, $("#college_search").val(), true);
            }
        });

        // Handle search functionality (minimum 3 characters)
        $("#college_search").on('input', function () {
            var searchValue = $(this).val().trim();

            clearTimeout(searchTimeout);

            // Auto-open dropdown when user starts typing (even before 3 chars)
            if (searchValue.length > 0) {
                $("#college_dropdown_header").addClass('active');
                $("#college_dropdown_list").addClass('active');
            }

            // Show visual feedback
            if (searchValue.length > 0 && searchValue.length < 3) {
                // Show hint that more characters are needed
                $("#college_dropdown_list").html('<div class="college-dropdown-item placeholder">Type at least 3 characters to search...</div>');
                $("#college_count_display").hide();
                return;
            }

            if (searchValue.length >= 3 || searchValue.length === 0) {
                searchTimeout = setTimeout(function () {
                    if (currentStateId) {
                        currentPage = 1;
                        hasMore = true;
                        console.log('Searching for:', searchValue);
                        loadColleges(currentStateId, currentPage, searchValue, false);
                    }
                }, 300); // Debounce search by 300ms
            }
        });

        // Prevent dropdown from closing when clicking search box
        $("#college_search").on('click', function (e) {
            e.stopPropagation();
        });
    }

    // Initialize when document is ready
    $(document).ready(function () {
        if ($("#college_state").length && $("#custom_college_dropdown_wrapper").length) {
            initCustomCollegeDropdown();
            console.log('Custom college dropdown initialized');
        }
    });

})(jQuery);

