const express = require('express');
const router = express.Router();
const { collegeQuery } = require('../config/db');
const auth = require('./auth');

// save new college
router.post('/new-college', auth, async (req, res) => {
    try {
        const { college_name, university_name, district_id, state_id, address } = req.body;
        const [result] = await collegeQuery('INSERT INTO new_colleges (college_name, university_name, district_id, state_id, address,created_by) VALUES (?, ?, ?, ?, ?, ?)', [college_name, university_name, district_id, state_id, address, req.name]);
        res.json({ success: true, college_id: result.insertId, message: 'College saved successfully' });
    } catch (error) {
        console.error('Error saving college:', error);
        res.status(500).json({ success: false, message: 'Failed to save college', error: error.message });
    }
});
// Get all states
router.get('/states', async (req, res) => {
    try {
        const states = await collegeQuery('SELECT id, name as state_name FROM states ORDER BY name ASC');
        res.json({
            success: true,
            error: 0,
            data: states
        });
    } catch (error) {
        console.error('Error fetching states:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to fetch states',
            error: error.message
        });
    }
});

// get courses
router.get('/courses', async (req, res) => {
    try {
        const courses = [
            {
                "id": "1",
                "name": "Embedded Systems",
                "active": "Yes"
            },
            {
                "id": "2",
                "name": "Full Stack Development",
                "active": "Yes"
            }
        ];
        res.json({
            success: true,
            error: 0,
            data: courses
        });
    } catch (error) {
        console.error('Error fetching courses:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to fetch courses',
            error: error.message
        });
    }
});

// get degrees
router.get('/degrees', async (req, res) => {
    try {
        const degrees = [
            {
                "id": "1",
                "name": "B.TECH",
                "pgugothers": "UG",
                "active": "Yes"
            },
            {
                "id": "2",
                "name": "BE",
                "pgugothers": "UG",
                "active": "Yes"
            },
            {
                "id": "3",
                "name": "M.TECH",
                "pgugothers": "PG",
                "active": "Yes"
            },
            {
                "id": "4",
                "name": "ME",
                "pgugothers": "PG",
                "active": "Yes"
            },
            {
                "id": "5",
                "name": "MS",
                "pgugothers": "PG",
                "active": "Yes"
            },
            {
                "id": "12",
                "name": "OTHERS",
                "pgugothers": "OTHERS",
                "active": "Yes"
            }
        ];
        res.json({
            success: true,
            error: 0,
            data: degrees
        });
    } catch (error) {
        console.error('Error fetching degrees:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to fetch degrees',
            error: error.message
        });
    }
});


// Get districts by state
router.get('/districts/:stateId', async (req, res) => {
    try {
        const districts = await collegeQuery('SELECT id, name FROM districts WHERE state_id = ? ORDER BY name ASC', [req.params.stateId]);
        res.json({
            success: true,
            districts: districts
        });
    } catch (error) {
        console.error('Error fetching districts:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to fetch districts',
            error: error.message
        });
    }
});

// Get colleges by state
router.get('/colleges/:stateId', async (req, res) => {
    try {
        const { page = 1, limit = 100, search = '' } = req.query;
        const offset = (parseInt(page) - 1) * parseInt(limit);

        // Sanitize and prepare search parameters
        let searchTerm = search.trim();
        // replace spaces with %
        searchTerm = searchTerm.replace(/[^a-zA-Z0-9]/g, '%');

        let query, queryParams;

        if (searchTerm) {
            // If search term exists, use it in the query
            query = `
                SELECT id, college_name, college_code, univerisity_name, college_address, college_website 
                FROM collegedirectory 
                WHERE state_id = ? 
                AND (
                    college_name LIKE ? 
                    OR univerisity_name LIKE ? 
                    OR college_address LIKE ?
                    OR college_website LIKE ?
                )
                ORDER BY 
                    CASE 
                        WHEN college_name LIKE ? THEN 1
                        WHEN college_name LIKE ? THEN 2
                        ELSE 3 
                    END,
                    college_name ASC 
                LIMIT ? OFFSET ?
            `;

            const searchPattern = `%${searchTerm}%`;
            const startPattern = `${searchTerm}%`;
            queryParams = [
                parseInt(req.params.stateId),
                searchPattern,
                searchPattern,
                searchPattern,
                searchPattern,
                startPattern,
                searchPattern,
                parseInt(limit),
                parseInt(offset)
            ];

            // Count query with search
            const countQuery = `
                SELECT CAST(COUNT(*) AS SIGNED) as total 
                FROM collegedirectory 
                WHERE state_id = ? 
                AND (
                    college_name LIKE ? 
                    OR univerisity_name LIKE ? 
                    OR college_address LIKE ?
                )
            `;
            const countParams = [
                parseInt(req.params.stateId),
                searchPattern,
                searchPattern,
                searchPattern
            ];

            const [countResult] = await collegeQuery(countQuery, countParams);
            const colleges = await collegeQuery(query, queryParams);

            return res.json({
                success: true,
                error: 0,
                data: colleges,
                colleges: colleges,
                pagination: {
                    total: Number(countResult.total),
                    page: parseInt(page),
                    limit: parseInt(limit),
                    totalPages: Math.ceil(Number(countResult.total) / parseInt(limit))
                }
            });
        } else {
            // If no search term, return all colleges for the state
            query = `
                SELECT id, college_name, college_code, univerisity_name, college_address, college_website 
                FROM collegedirectory 
                WHERE state_id = ? 
                ORDER BY college_name ASC 
                LIMIT ? OFFSET ?
            `;
            queryParams = [parseInt(req.params.stateId), parseInt(limit), parseInt(offset)];

            // Simple count query with explicit cast to signed integer
            const [countResult] = await collegeQuery(
                'SELECT CAST(COUNT(*) AS SIGNED) as total FROM collegedirectory WHERE state_id = ?',
                [parseInt(req.params.stateId)]
            );

            const colleges = await collegeQuery(query, queryParams);
            return res.json({
                success: true,
                error: 0,
                data: colleges,
                colleges: colleges,
                pagination: {
                    total: Number(countResult.total),
                    page: parseInt(page),
                    limit: parseInt(limit),
                    totalPages: Math.ceil(Number(countResult.total) / parseInt(limit))
                }
            });
        }
    } catch (error) {
        console.error('Error fetching colleges:', error);
        return res.status(500).json({
            success: false,
            message: 'Failed to fetch colleges',
            error: error.message
        });
    }
});

module.exports = router;