const express = require('express');
const router = express.Router();
const { vtdbpool } = require('../config/db');
const axios = require('axios');
const auth = require('./auth');

// Constants
const OTP_EXPIRY_MINUTES = 95;
const MAX_OTP_ATTEMPTS = 5;
const SMS_API_KEY = process.env.SMS_API_KEY;
const SMS_SENDER = process.env.SMS_SENDER;

// Generate a random 6-digit OTP
const generateOTP = () => {
    return Math.floor(100000 + Math.random() * 900000).toString();
};

// Check SMS status
const checkSMSStatus = async (messageId) => {
    try {
        //http://www.bulksmsapps.com/api/apiDeliveryStatusv2.aspx\?apikey\=401e5003-e001-470f-b320-742b46e8a607\&messageid\=259177113
        const url = `http://www.bulksmsapps.com/api/apiDeliveryStatusv2.aspx?apikey=${SMS_API_KEY}&messageid=${messageId}`;
        const response = await axios.get(url);
        let status = '';
        console.log("smsResponse:", response.data);
        const messageLine = response.data.split('\n')[0].trim();
        console.log("messageLine:", messageLine);
        const details = messageLine.split('-');
        console.log("details:", details);
        if (details.length > 1) {
            if (details[1] === 'DELIVRD') {
                status = 'delivered';
            } else if (details[1] === 'EXPIRED') {
                status = 'failed';
            } else if (details[1] === 'UNDELIV') {
                status = 'failed';
            } else if (details[1] === 'FAILED') {
                status = 'failed';
            } else if (details[1] === 'REJECTD') {
                status = 'failed';
            } else if (details[1] === 'AWAITED') {
                status = 'accepted';
            } else if (details[1] === 'pending') {
                status = 'accepted';
            } else if (details[1] === 'sent') {
                status = 'sent';
            } else if (details[1] === 'delivered') {
                status = 'delivered';
            } else if (details[1] === 'received') {
                status = 'received';
            } else if (details[1] === 'failed') {
                status = 'failed';
            }
        }
        return { success: true, response: response.data, status: status };
    } catch (error) {
        console.error("Check SMS status error:", error);
        return { success: false, error: error.message };
    }
}
// Send OTP via SMS
const sendSMS = async (mobile, otp) => {
    try {
        const message = `Your OTP for verify mobile number in Registration is ${otp}. --  Vector India`;
        const url = `http://www.bulksmsapps.com/api/apismsv2.aspx?apikey=${SMS_API_KEY}&sender=${SMS_SENDER}&number=${mobile}&message=${encodeURIComponent(message)}`;

        const response = await axios.get(url);
        console.log("response:", response.data);
        const messageLine = response.data.split('\n')[0];
        console.log("messageLine:", messageLine);
        const messageId = messageLine.split('-')[1]?.trim();
        console.log("messageId:", messageId);
        if (response.status === 200) {
            return { success: true, response: response.data, messageId: messageId };
        } else {
            throw new Error('SMS API returned error: ' + response.data);
        }
    } catch (error) {
        console.error('SMS sending error:', error);
        return { success: false, error: error.message };
    }
};

// Send OTP via WhatsApp (you'll need to implement this with your WhatsApp Business API)
const sendWhatsApp = async (mobile, otp, user_id) => {
    try {

        const url = `https://cv.vectorindia.org/cvapi/whatsapp/webhook/verifyotp`;
        console.log(`Sending WhatsApp OTP to ${mobile}: ${otp}`);
        const response = await axios({
            method: 'post',
            url: url,
            data: {
                phoneNumber: mobile,
                otp: otp,
                user_id: user_id
            }
        });
        console.log(response.data);
        return { success: true };
    } catch (error) {
        console.error('WhatsApp sending error:', error);
        return { success: false, error: error.message };
    }
};

// Validate mobile number
const validateMobile = (mobile) => {
    const mobileRegex = /^[6-9]\d{9}$/;
    return mobileRegex.test(mobile);
};

// Send OTP
router.post('/send-otp', auth, async (req, res) => {
    const conn = await vtdbpool.getConnection();

    try {
        const { mobile, method = 'sms' } = req.body;

        // Validate input
        if (!mobile) {
            return res.status(400).json({
                success: false,
                message: 'Mobile number is required'
            });
        }

        if (!validateMobile(mobile)) {
            return res.status(400).json({
                success: false,
                message: 'Invalid mobile number format'
            });
        }

        // Check if mobile is already verified
        const verifiedMobile = await conn.query(
            'SELECT * FROM verification_otps WHERE mobile = ? AND verified = true',
            [mobile]
        );
        console.log("verifiedMobile:", verifiedMobile);

        if (verifiedMobile.length > 0) {
            // Get user details if available
            const userDetails = await conn.query(
                'SELECT * FROM leads WHERE mobile = ? LIMIT 1',
                [mobile]
            );
            let mobileVerified = false;
            let whatsappVerified = false;
            for (verified of verifiedMobile) {
                if (verified.method === 'sms') {
                    mobileVerified = true;
                } else if (verified.method === 'whatsapp') {
                    whatsappVerified = true;
                }
            }

            return res.status(200).json({
                success: true,
                verified: true,
                message: 'Mobile number is already verified',
                user: userDetails.length > 0 ? userDetails[0] : null,
                mobileVerified: mobileVerified,
                whatsappVerified: whatsappVerified
            });
        }

        // Check for recent OTP attempts
        const recentOtp = await conn.query(
            `SELECT COUNT(*) as count FROM verification_otps 
             WHERE mobile = ? and method = ? AND created_at > NOW() - INTERVAL ? MINUTE`,
            [mobile, method, OTP_EXPIRY_MINUTES]
        );

        console.log("recentOtp:", recentOtp);
        if (recentOtp.length > 0) {
            if (recentOtp[0].count >= MAX_OTP_ATTEMPTS) {
                return res.status(429).json({
                    success: false,
                    message: `Too many OTP attempts. Please try after ${OTP_EXPIRY_MINUTES} minutes`
                });
            }
        }

        // Generate and save new OTP
        const otp = generateOTP();
        const hashId = require('crypto').randomBytes(16).toString('hex');

        // Save OTP to database
        await conn.query(
            `INSERT INTO verification_otps (hash_id, mobile, otp, method, created_at) 
             VALUES (?, ?, ?, ?, NOW())`,
            [hashId, mobile, otp, method]
        );

        // Send OTP based on method
        const sendResult = method === 'whatsapp'
            ? await sendWhatsApp(mobile, otp, req.user)
            : await sendSMS(mobile, otp);

        if (!sendResult.success) {
            throw new Error(`Failed to send OTP via ${method}: ${sendResult.error}`);
        }

        res.json({
            success: true,
            message: `OTP sent successfully via ${method}`,
            hashId
        });

    } catch (error) {
        console.error('OTP sending error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to send OTP',
            error: error.message
        });
    } finally {
        conn.release();
    }
});

// Verify OTP
router.post('/verify-otp', auth, async (req, res) => {
    const conn = await vtdbpool.getConnection();

    try {
        const { mobile, otp, hashId } = req.body;

        // Validate input
        if (!mobile || !otp || !hashId) {
            return res.status(400).json({
                success: false,
                message: 'Mobile number, OTP, and hash ID are required'
            });
        }

        if (!validateMobile(mobile)) {
            return res.status(400).json({
                success: false,
                message: 'Invalid mobile number format'
            });
        }

        // Get OTP record
        const otpRecord = await conn.query(
            `SELECT * FROM verification_otps 
             WHERE mobile = ? AND hash_id = ? AND verified = false 
             AND created_at > NOW() - INTERVAL ? MINUTE
             ORDER BY created_at DESC LIMIT 1`,
            [mobile, hashId, OTP_EXPIRY_MINUTES]
        );
        console.log("otpRecord:", otpRecord);

        if (!otpRecord.length) {
            return res.status(404).json({
                success: false,
                message: 'OTP expired or not found. Please request a new OTP.'
            });
        }

        // Verify OTP
        if (otpRecord[0].otp !== otp) {
            // Increment failed attempts
            await conn.query(
                'UPDATE verification_otps SET failed_attempts = failed_attempts + 1 WHERE id = ?',
                [otpRecord[0].id]
            );

            return res.status(400).json({
                success: false,
                message: 'Invalid OTP. Please try again.'
            });
        }

        // Mark OTP as verified
        await conn.query(
            `UPDATE verification_otps 
             SET verified = true, verified_at = NOW() 
             WHERE id = ?`,
            [otpRecord[0].id]
        );

        // Update user's mobile verification status if needed
        await conn.query(
            `UPDATE leads 
             SET validate_mobile = true 
             WHERE mobile = ?`,
            [mobile]
        );

        res.json({
            success: true,
            message: 'Mobile number verified successfully'
        });

    } catch (error) {
        console.error('OTP verification error:', error);
        res.status(500).json({
            success: false,
            message: 'Failed to verify OTP',
            error: error.message
        });
    } finally {
        conn.release();
    }
});

// Export functions for use in other modules
module.exports = {
    router,
    generateOTP,
    checkSMSStatus,
    sendSMS,
    sendWhatsApp,
    validateMobile
}; 